use core:lang;
use core:asm;
use lang:bs:macro;
use lang:bs;

on Compiler:

void addGlobal(Content content, SrcPos pos, STypeName type, SStr name, SExpr? init) {
	content.add(CppGlobalScalarDecl(pos, type, name.v, init));
}

void addGlobalArray(Content content, SrcPos pos, STypeName type, SStr name, SExpr count, SArrayInit? init) {
	content.add(CppGlobalArrayDecl(pos, type, name.v, count, init));
}

void addGlobalArray(Content content, SrcPos pos, STypeName type, SStr name, SArrayInit init) {
	content.add(CppGlobalArrayDecl(pos, type, name.v, null, init));
}

/**
 * Declared global variables.
 */
class CppGlobalDecl on Compiler {
	// Position of the declaration.
	SrcPos pos;

	// Name of the variable.
	Str name;

	init(SrcPos pos, Str name) {
		init() {
			pos = pos;
			name = name;
		}
	}

	void create(Package pkg) : abstract;
}

/**
 * Declared global scalar variable.
 */
class CppGlobalScalarDecl extends CppGlobalDecl {
	// Name of the type.
	STypeName type;

	// Initializer, if present.
	SExpr? init;

	init(SrcPos pos, STypeName type, Str name, SExpr? init) {
		init(pos, name) {
			type = type;
			init = init;
		}
	}

	void create(Package pkg) {
		pkg.add(CppGlobal(this, cppScope(pkg)));
	}
}

/**
 * Declared global array variable.
 *
 * Note: at least one of 'count' or 'initializers' must be set.
 */
class CppGlobalArrayDecl extends CppGlobalDecl {
	// Name of the type.
	STypeName type;

	// Number of elements in the array.
	SExpr? count;

	// Initializer for the array.
	SArrayInit? initializers;

	init(SrcPos pos, STypeName type, Str name, SExpr? count, SArrayInit? init) {
		init(pos, name) {
			type = type;
			count = count;
			initializers = init;
		}
	}

	void create(Package pkg) {
		pkg.add(CppGlobalArray(this, cppScope(pkg)));
	}
}

/**
 * Global scalar variable.
 */
class CppGlobal extends GlobalVar {
	// Constant variable?
	Bool const;

	init(CppGlobalScalarDecl decl, Scope scope) {
		var type = decl.type.transform(scope);
		if (!type.isValue)
			throw SyntaxError(decl.pos, "Only value-types are supported. Did you declare this type in C or C++?");

		var thread = named{ui:Render};

		Function init = initFn(safeType(type, decl.pos), scope, decl.pos, decl.init);
		init.parentLookup = scope.top;
		init.runOn(thread);

		init(decl.name, type.asRef(false), thread, init.pointer()) {
			const = type.const;
		}
	}

	// Create a function that initializes the value.
	Function initFn(Type type, Scope scope, SrcPos pos, SExpr? init) : static {
		Listing l(false, type.typeDesc());

		var v = l.createVar(l.root, type.size);
		l << prolog();

		if (init) {
			// Initializer expression?
			DummyBlock b(pos, scope);
			Expr expr = init.transform(b);

			// Ensure that it is a constant expression. Otherwise, strange things may happen to the visualization!
			if (expr.staticValue(type).empty)
				throw SyntaxError(pos, "Initializers for global variables must be constant.");

			// Generate code for the expression.
			CodeGen gen(RunOn(), l);
			expr.code(gen, CodeResult(type, v));
		} else if (cpp = cppCtor(type, [], scope)) {
			// C++ type?
			l << lea(ptrA, v);
			l << fnParam(ptrDesc, ptrA);
			l << fnParam(intDesc, natConst(0));
			l << fnCall(cpp.ref, true);
		} else if (storm = stormCtor(type, [], scope)) {
			l << lea(ptrA, v);
			l << fnParam(ptrDesc, ptrA);
			l << fnCall(storm.ref, true);
		}

		l << fnRet(v);

		Function fn(Value(type), "init", []);
		fn.setCode(DynamicCode(l));
		fn;
	}
}

/**
 * Global array variable.
 *
 * Note: We store these as if they are pointers to the allocation. We can't allocate an array of
 * elements as we do for local array variables, since we use the standard GlobalVar here. This has
 * implications for the visualization, but it is likely a good thing as it reduces the size of the
 * Globals box.
 */
class CppGlobalArray extends GlobalVar {
	init(CppGlobalArrayDecl decl, Scope scope) {
		var type = decl.type.transform(scope);
		if (!type.isValue)
			throw SyntaxError(decl.pos, "Only value-types are supported. Did you declare this type in C or C++?");
		Value ptrType = if (type.const) { wrapConstPtr(type); } else { wrapPtr(type); };

		var thread = named{ui:Render};

		Function init = initFn(safeType(ptrType, decl.pos), safeType(type, decl.pos), scope, decl.pos, decl.count, decl.initializers);
		init.parentLookup = scope.top;
		init.runOn(thread);

		init(decl.name, ptrType, thread, init.pointer()) {}
	}

	// Create a function that initializes the value.
	Function initFn(Type ptrType, Type type, Scope scope, SrcPos pos, SExpr? count, SArrayInit? init) : static {
		DummyBlock block(pos, scope);

		Array<Expr> initExprs;
		if (init) {
			initExprs = init.transform(block);
			for (i, x in initExprs) {
				if (x.staticValue(type).empty)
					throw SyntaxError(x.pos, "This expression is not constant, and can not be used to initialize a global array.");
			}
		}

		Nat arrayCount = 0;
		if (count) {
			Expr countExpr = count.transform(block);
			if (c = variantToInt(countExpr.staticValue())) {
				if (c < 0)
					throw SyntaxError(pos, "Array size can not be negative (it is ${c})");
				arrayCount = c.nat;
			} else {
				throw SyntaxError(pos, "Array size of global arrays must be constant.");
			}

			if (init) {
				if (initExprs.count > arrayCount)
					throw SyntaxError(pos, "Too many initializers. The array has size ${arrayCount} but ${initExprs.count} initializers.");

				while (initExprs.count < arrayCount)
					initExprs << defCtorCall(pos, type, scope);
			}
		} else if (init) {
			arrayCount = initExprs.count;
		} else {
			// Should not happen if other code in the frontend works properly.
			throw SyntaxError(pos, "Either array count or initializer must be specified.");
		}

		Listing l(false, ptrType.typeDesc());

		var v = l.createVar(l.root, ptrType.size);

		l << prolog();

		// Allocate memory.
		l << fnParam(ptrDesc, type.typeRef);
		l << fnParam(ptrDesc, ptrConst(arrayCount));
		l << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, ptrA);

		// Save the pointer to the allocation in 'v'.
		l << mov(ptrRel(v, Offset()), ptrA);
		// Initialize the offset.
		l << mov(intRel(v, Offset(sPtr)), natConst(sPtr * 2));

		// Set 'filled'.
		Nat mask = AllocFlags:arrayAlloc.v;
		l << mov(intRel(ptrA, Offset(sPtr)), natConst(mask | arrayCount));

		if (init.any) {
			// Use initializer expressions.
			CodeGen gen(RunOn(), l);
			Value valType(type);
			Size sz = valType.size.aligned;
			for (i, x in initExprs) {
				CodeResult result(type, gen.block);
				x.code(gen, result);
				result.created(gen);

				// Copy it to the appropriate place. Since it is a statically known expression, we
				// assume that the expression is copiable through Storm mechanisms. I.e. we don't
				// bother with C++-style copy-ctors.
				l << mov(ptrA, ptrRel(v, Offset()));
				l << add(ptrA, ptrConst(sPtr*2 + sz*i));
				if (valType.isAsmType) {
					l << mov(xRel(sz, ptrA, Offset()), result.location(gen));
				} else {
					l << lea(ptrC, result.location(gen));
					l << fnParam(ptrDesc, ptrA);
					l << fnParam(ptrDesc, ptrC);
					l << fnCall(valType.copyCtor, true);
				}
			}
		} else {
			// Use standard constructor. Emit a loop.
			var loopTop = l.label();
			var loopEnd = l.label();

			// Note: default-initialized to zero.
			var counter = l.createVar(l.root, sInt);
			l << loopTop;
			l << cmp(counter, natConst(arrayCount));
			l << jmp(loopEnd, CondFlag:ifAboveEqual);

			// Compute the offset from the start of the array.
			l << icast(ptrA, counter);
			l << mul(ptrA, ptrConst(type.size.aligned));
			l << add(ptrA, ptrConst(sPtr*2));

			if (cpp = cppCtor(type, [], scope)) {
				// If it is a C++ constructor, pass the base and the offset to it.
				l << fnParam(ptrDesc, ptrRel(v, Offset()));
				l << fnParam(intDesc, eax); // implicit cast to nat
				l << fnCall(cpp.ref, true);
			} else if (storm = stormCtor(type, [], scope)) {
				// If it is a Storm constructor, pass a proper pointer instead.
				l << add(ptrA, ptrRel(v, Offset()));
				l << fnParam(ptrDesc, ptrA);
				l << fnCall(storm.ref, true);
			}

			l << add(counter, natConst(1));
			l << jmp(loopTop);

			l << loopEnd;
		}

		l << fnRet(v);

		Function fn(Value(ptrType), "init", []);
		fn.setCode(DynamicCode(l));
		fn;
	}
}

/**
 * Access a global variable.
 */
class GlobalVarAccess extends Expr {
	// Variable to access.
	GlobalVar var;

	init(SrcPos pos, GlobalVar var) {
		init(pos) { var = var; }
	}

	Bool temporary() : override {
		false;
	}

	Bool constant() : override {
		// If it is an array, act as if the pointer was a temporary. Otherwise it would be possible
		// to modify what the global refers to!
		if (var as CppGlobalArray)
			return true;
		if (var as CppGlobal)
			return var.const;
		false;
	}

	ExprResult result() : override {
		var.type.asRef();
	}

	Variant staticValue(Type t) : override {
		if (!constant)
			return Variant();

		if (Value(t).mayReferTo(var.type))
			return var.value;

		return Variant();
	}

	void code(CodeGen gen, CodeResult result) : override {
		if (!result.needed())
			return;

		gen.l << fnParam(ptrDesc, objPtr(var));
		gen.l << fnCall(ref(BuiltIn:globalAddr), true, ptrDesc, ptrB);

		if (result.type.ref) {
			gen.l << mov(result.location(gen), ptrB);
		} else {
			var r = result.location(gen);
			if (result.type.isAsmType) {
				gen.l << mov(r, xRel(r.size, ptrB));
			} else {
				gen.l << lea(ptrA, r);
				gen.l << fnParam(ptrDesc, ptrA);
				gen.l << fnParam(ptrDesc, ptrB);
				gen.l << fnCall(result.type.copyCtor, true);
			}
		}
	}

	void codePtr(CodeGen gen, Type type) : override {
		gen.l << fnParam(ptrDesc, objPtr(var));
		gen.l << fnCall(ref(BuiltIn:globalAddr), true, ptrDesc, ptrB);

		if (var.type.isValue) {
			// Globals are stored in an array allocation, exactly how we're doing it.
			gen.l << sub(ptrB, ptrConst(sPtr * 2));
			gen.l << mov(ecx, natConst(sPtr * 2));
		} else {
			throw SyntaxError(pos, "Reading non-value globals is not yet supported. Did you declare this variable in C++?");
		}
	}
}
