/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * gui initialization and top-level event handling stuff
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/time.h>
#include <errno.h>
#include <pthread.h>
#include <signal.h>
#include <limits.h>
#include <zlib.h>
#include <sys/wait.h>
#include <time.h>
#include <stdlib.h>
#include <unistd.h>

/* input_pvr functionality needs this */
#define XINE_ENABLE_EXPERIMENTAL_FEATURES

#include "common.h"
#include "kbindings.h"
#include "config_wrapper.h"
#include "event.h"
#include "actions.h"
#include "acontrol.h"
#include "control.h"
#include "event_sender.h"
#include "help.h"
#include "mediamark.h"
#include "menus.h"
#include "mrl_browser.h"
#include "osd.h"
#include "panel.h"
#include "playlist.h"
#include "post.h"
#include "setup.h"
#include "stream_infos.h"
#include "videowin.h"
#include "viewlog.h"
#include "skins.h"
#include "session.h"
#include "network.h"
#include "tvout.h"
#include "tvset.h"
#include "stdctl.h"
#include "lirc.h"
#include "oxine/oxine.h"
#include "errors.h"

/*
 * global variables
 */
static pid_t            xine_pid;

/* Icon data */
static const unsigned char icon_datas[] = {
   0x11, 0x00, 0x00, 0x00, 0x88, 0x11, 0x00, 0x00, 0x00, 0x88, 0xff, 0xff,
   0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf1, 0xff, 0xff, 0xff,
   0x8f, 0xf1, 0xff, 0xff, 0xff, 0x8f, 0x1f, 0x00, 0x00, 0x00, 0xf8, 0x1f,
   0x00, 0x00, 0x00, 0xf8, 0x11, 0x00, 0x00, 0x00, 0x88, 0x11, 0x00, 0x00,
   0x00, 0x88, 0x1f, 0x00, 0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0x00, 0xf8,
   0x11, 0x00, 0x00, 0x00, 0x88, 0x11, 0x00, 0x00, 0x00, 0x88, 0x1f, 0x00,
   0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0x00, 0xf8, 0x11, 0x00, 0x00, 0x00,
   0x88, 0x11, 0x91, 0x79, 0xf8, 0x88, 0x1f, 0x9b, 0x99, 0x0c, 0xf8, 0x1f,
   0x8e, 0x99, 0x7d, 0xf8, 0x11, 0x8e, 0x99, 0x7d, 0x88, 0x11, 0x9b, 0x99,
   0x0d, 0x88, 0x1f, 0x91, 0x99, 0xf9, 0xf8, 0x1f, 0x00, 0x00, 0x00, 0xf8,
   0x11, 0x00, 0x00, 0x00, 0x88, 0x11, 0x00, 0x00, 0x00, 0x88, 0x1f, 0x00,
   0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0x00, 0xf8, 0x11, 0x00, 0x00, 0x00,
   0x88, 0x11, 0x00, 0x00, 0x00, 0x88, 0x1f, 0x00, 0x00, 0x00, 0xf8, 0x1f,
   0x00, 0x00, 0x00, 0xf8, 0xf1, 0xff, 0xff, 0xff, 0x8f, 0xf1, 0xff, 0xff,
   0xff, 0x8f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
   0x11, 0x00, 0x00, 0x00, 0x88, 0x11, 0x00, 0x00, 0x00, 0x88, 0x1f, 0x00,
   0x00, 0x00, 0xf8, 0x1f, 0x00, 0x00, 0x00, 0xf8
};

static const char *const exp_levels[] = {
  "Beginner",
  "Advanced",
  "Expert",
  "Master of the known universe",
#ifdef DEBUG
  "Debugger",
#else
  NULL,
#endif
  NULL
};

static const char *const visual_anim_style[] = {
  "None",
  "Post Plugin",
  "Stream Animation",
  NULL
};

static const char *const mixer_control_method[] = {
  "Sound card",
  "Software",
  NULL
};

static const char *const shortcut_style[] = {
  "Windows style",
  "Emacs style",
  NULL
};

void dummy_config_cb(void *data, xine_cfg_entry_t *cfg) {
  /* It exist to avoid "restart" window message in setup window */
  (void)data;
  (void)cfg;
}
static void auto_vo_visibility_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_auto_vo_vis;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_auto_vo_vis);
}
static void auto_panel_visibility_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_auto_panel_vis;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_auto_panel_vis);
}
static void skip_by_chapter_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->skip_by_chapter = cfg->num_value;
  panel_update_nextprev_tips (gui->panel);
}
static void ssaver_timeout_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->ssaver_timeout = cfg->num_value;
}

static void visual_anim_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  /* something to do? */
  if (gui->visual_anim.enabled == cfg->num_value)
    return;
  /* stop old anim. */
  if (gui->visual_anim.enabled == 1) {
    if (post_rewire_audio_port_to_stream (gui, gui->stream))
      gui->visual_anim.running = 0;
  } else if (gui->visual_anim.enabled == 2) {
    visual_anim_play (gui, 0);
  }
  /* set new mode. */
  gui->visual_anim.enabled = cfg->num_value;

  /* NOTE: freeze alert:
   * 1. This runs with xine config locked.
   * 2. xine_open () will wait for xine video decoder thread to load
   *    plugins, which also wants to lock xine config.
   * Better defer this to next stream (or maybe to our slider thread?). */
  if (cfg->num_value == 2) {
    /*
    if (!has_video)
        visual_anim_play (gui, 1);
    */
    return;
  }

  if (gui->visual_anim.enabled == 1) {
    int has_video =  xine_get_stream_info (gui->stream, XINE_STREAM_INFO_HAS_VIDEO)
                 && !xine_get_stream_info (gui->stream, XINE_STREAM_INFO_IGNORE_VIDEO);

    if (!has_video
      && gui->visual_anim.post_output_element.post
      && post_rewire_audio_post_to_stream (gui, gui->stream))
        gui->visual_anim.running = 1;
  }
}

static void stream_info_auto_update_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_sinfo_update;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_sinfo_update);
  stream_infos_toggle_auto_update (gui->streaminfo);
}

/*
 * Layer above callbacks
 */
static void layer_above_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->layer_above &= ~1;
  gui->layer_above |= cfg->num_value ? 1 : 0;
}
static void always_layer_above_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->layer_above &= ~2;
  gui->layer_above |= cfg->num_value ? 2 : 0;
}

/*
 * Callback for snapshots saving location.
 */
static void snapshot_loc_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->snapshot_location = cfg->str_value;
}

/*
 * Callback for skin server
 */
static void skin_server_url_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  free (gui->skin_server_url);
  gui->skin_server_url = strdup (cfg->str_value);
}

static void smart_mode_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_smart_mode;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_smart_mode);
}

static void play_anyway_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->flags &= ~XUI_FLAG_play_anyway;
  gui->flags |= xitk_bitmove (cfg->num_value, 1, XUI_FLAG_play_anyway);
}

static void exp_level_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->experience_level = (cfg->num_value * 10);
}

static void audio_mixer_method_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  xui_mixer_type_t type = cfg->num_value;
  if (type >= LAST_MIXER)
    type = SOFTWARE_MIXER;
  gui->mixer.type_volume = type;
  gui->mixer.type_mute = gui->mixer.mute[type] >= 0 ? type : SOFTWARE_MIXER;
  panel_update_mixer_display (gui->panel);
}

static void shortcut_style_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->shortcut_style = cfg->num_value;
}

static void implicit_aliases_cb(void *data, xine_cfg_entry_t *cfg) {
  gGui_t *gui = data;
  gui->kbindings_enabled = (gui->kbindings_enabled & ~2) | xitk_bitmove (cfg->num_value, 1, 2);
}

/*
 *
 */
static int gui_signal_handler (int sig, void *data) {
  pid_t cur_pid = getppid ();
  gGui_t *gui = data;

  (void)data;
  if (cur_pid != xine_pid)
    return 0;

  switch (sig) {
    case SIGHUP:
      printf ("gui: SIGHUP received: re-read config file.\n");
      xine_config_reset (gui->xine);
      xine_config_load (gui->xine, gui->cfg_file);
      return 1;
    case SIGUSR1:
      printf ("gui: SIGUSR1 received.\n");
      return 1;
    case SIGUSR2:
      printf ("gui: SIGUSR2 received.\n");
      return 1;
    case SIGINT:
    case SIGTERM:
      printf ("gui: SIGINT/SIGTERM received.\n");
      /* xine_config_save (gui->xine, gui->cfg_file); */
      gui_exit (NULL, gui);
      return 1;
    default: ;
  }
  return 0;
}


/*
 * convert pts value to string
 */
static const char *pts2str (char *buf, int pts) {
  char *q = buf + 40;
  uint32_t minus, val, v2;

  if (pts < 0)
    minus = 1, val = -pts;
  else
    minus = 0, val = pts;

  q -= 6;
  memcpy (q, " pts)", 6);
  v2 = val;
  do {
    *--q = '0' + (v2 % 10u);
    v2 /= 10u;
  } while (v2);
  if (minus)
    *--q = '-';
  q -= 2;
  memcpy (q, " (", 2);

  val /= 900;
  v2 = val % 6000; /* 1/100s */
  val /= 6000; /* mins */
  *--q = '0' + (v2 % 10u); v2 /= 10u;
  *--q = '0' + (v2 % 10u); v2 /= 10u;
  *--q = '.';
  *--q = '0' + (v2 % 10u); v2 /= 10u;
  *--q = '0' + v2;
  *--q = ':';
  do {
    *--q = '0' + (val % 10u);
    val /= 10u;
  } while (val);
  if (minus)
    *--q = '-';

  return q;
}

static ATTR_FORMAT_ARG (1) const char *_silence_nonliteral_fmt_warn (const char *test, const char *s) {
  (void)test;
  return s;
}

/*
 *
 */
void gui_action_args (gGui_t *gui, action_id_t id, int num_arg, const char *str_arg) {
  xine_event_t xine_event;
  char buf[40];
  int val;

  if (gui->verbosity >= 2)
    printf ("gui.action.call (%d, %s)%s.\n",
      (int)id,  kbindings_action_name (gui->kbindings, id), gui->event_reject ? " [rejected]" : "");
  if (gui->event_reject)
    return;

  pthread_mutex_lock (&gui->event_mutex);

  if (gui->event_reject) {
    pthread_mutex_unlock (&gui->event_mutex);
    return;
  }

  gui->event_pending++;

  if (id & ACTID_IS_INPUT_EVENT) {
    val = id;
    val -= ACTID_EVENT_NUMBER_0;

    /* Note: In the following overflow checks, we must test against INT_MAX */
    /* carefully. Otherwise, the comparison term may overflow itself and    */
    /* detecting the overflow condition will fail (never true or true by    */
    /* chance, depending on expression and rearranging by the compiler).    */

    if (XITK_0_TO_MAX_MINUS_1 (val, 10)) {
      if (gui->action_num_arg == XITK_INT_KEEP) {
        gui->action_num_arg = val;
      } else {
        unsigned int v = gui->action_num_arg;
        v = 10u * v + val;
        if (v <= INT_MAX)
          gui->action_num_arg = v;
        else
          fprintf (stderr, "xine-ui: Input number overflow, using %d\n", gui->action_num_arg);
      }
    } else if (id == ACTID_EVENT_NUMBER_10_ADD) {
      if (gui->action_num_arg == XITK_INT_KEEP)
        gui->action_num_arg = 10;
      else if (gui->action_num_arg <= (INT_MAX - 10))
        gui->action_num_arg += 10;
      else
        fprintf (stderr, "xine-ui: Input number overflow, using %d\n", gui->action_num_arg);
    } else {
      gui->action_num_arg = XITK_INT_KEEP;
    }
    num_arg = gui->action_num_arg;

    pthread_mutex_unlock (&gui->event_mutex);

    /* check if osd menu like this event */
    if (oxine_action_event (gui->oxine, id & ~ACTID_IS_INPUT_EVENT)) {
      pthread_mutex_lock (&gui->event_mutex);
      gui->event_pending--;
      if ((gui->event_pending <= 0) && gui->event_reject)
        pthread_cond_signal (&gui->event_safe);
      pthread_mutex_unlock (&gui->event_mutex);
      return;
    }

    /* events for advanced input plugins. */
    xine_event.type        = id & ~ACTID_IS_INPUT_EVENT;
    xine_event.data_length = 0;
    xine_event.data        = NULL;
    xine_event.stream      = gui->stream;
    xitk_gettime_tv (&xine_event.tv);

    xine_event_send(gui->stream, &xine_event);

    if (num_arg != XITK_INT_KEEP)
      osd_message (gui, "> %d_", num_arg);

    pthread_mutex_lock (&gui->event_mutex);
    gui->event_pending--;
    if ((gui->event_pending <= 0) && gui->event_reject)
      pthread_cond_signal (&gui->event_safe);
    pthread_mutex_unlock (&gui->event_mutex);
    return;
  }

  if (num_arg == XITK_INT_KEEP)
    num_arg = gui->action_num_arg;
  gui->action_num_arg = XITK_INT_KEEP;

  pthread_mutex_unlock (&gui->event_mutex);

  switch (id) {

  case ACTID_WINDOWREDUCE:
  case ACTID_WINDOWENLARGE:
  case ACTID_ZOOM_1_1:
  case ACTID_WINDOW100:
  case ACTID_WINDOW200:
  case ACTID_WINDOW50:
    {
      static const vwin_mag_t mags[] = {
        /* ACTID_WINDOWREDUCE  */ -0.840896415254 * VWIN_MAG_1, /* 1 / (2 ^ (1/4)) */
        /* ACTID_WINDOWENLARGE */ -1.189207115 * VWIN_MAG_1, /* 2 ^ (1/4) */
        /* ACTID_ZOOM_1_1 */       1 * VWIN_MAG_1,
        /* ACTID_WINDOW100 */      1 * VWIN_MAG_1,
        /* ACTID_WINDOW200 */      2 * VWIN_MAG_1,
        /* ACTID_WINDOW50 */       0.5 * VWIN_MAG_1
      };
        static const char *msgs [] = {
        /* ACTID_WINDOWREDUCE  */ N_("Zoom: %d%%"),
        /* ACTID_WINDOWENLARGE */ N_("Zoom: %d%%"),
        /* ACTID_ZOOM_1_1 */      N_("Zoom: 1:1"),
        /* ACTID_WINDOW100 */     N_("Zoom: 1:1"),
        /* ACTID_WINDOW200 */     N_("Zoom: 200%%"),
        /* ACTID_WINDOW50 */      N_("Zoom: 50%%")
      };
      int mag = video_window_set_mag (gui->vwin, mags[id - ACTID_WINDOWREDUCE]);
      /* i guess it is safe to supply an arg that some format strings will ignore. */
      if (mag)
        osd_message (gui, _silence_nonliteral_fmt_warn ("Zoom: %d%%",
          gettext (msgs[id - ACTID_WINDOWREDUCE])), (mag * 100 + (VWIN_MAG_1 >> 1)) / VWIN_MAG_1);
    }
    break;

  case ACTID_SPU_NEXT:
    if (num_arg == XITK_INT_KEEP)
      gui_nextprev_spu_channel (NULL, GUI_NEXT (gui));
    else
      gui_direct_change_spu_channel (NULL, gui, num_arg);
    break;

  case ACTID_SPU_PRIOR:
    if (num_arg == XITK_INT_KEEP)
      gui_nextprev_spu_channel (NULL, GUI_PREV (gui));
    else
      gui_direct_change_spu_channel (NULL, gui, num_arg);
    break;

  case ACTID_CONTROLSHOW:
    control_main (NULL, gui);
    break;

  case ACTID_ACONTROLSHOW:
    acontrol_main (NULL, gui);
    break;

  case ACTID_TOGGLE_WINOUT_VISIBLITY:
    if (!gui->use_root_window) {
      int visible = video_window_is_visible (gui->vwin) < 2;

      video_window_set_visibility (gui->vwin, visible);
      /* (re)start/stop visual animation */
      if (video_window_is_visible (gui->vwin) > 1) {
        if (gui->visual_anim.enabled && (gui->visual_anim.running == 2))
          visual_anim_play (gui, 1);
      } else {
        if (gui->visual_anim.running) {
          visual_anim_play (gui, 0);
          gui->visual_anim.running = 2;
        }
      }
    }
    break;

  case ACTID_TOGGLE_WINOUT_BORDER:
    video_window_mode (gui->vwin, TOGGLE_MODE | BORDER_MODE);
    break;

  case ACTID_NET_REMOTE:
    gui->network = !gui->network;
    stop_remote_server (gui);
    start_remote_server (gui);
    break;

  case ACTID_AUDIOCHAN_NEXT:
    if (num_arg == XITK_INT_KEEP)
      gui_nextprev_audio_channel (NULL, GUI_NEXT (gui));
    else
      gui_direct_change_audio_channel (NULL, gui, num_arg);
    break;

  case ACTID_AUDIOCHAN_PRIOR:
    if (num_arg == XITK_INT_KEEP)
      gui_nextprev_audio_channel (NULL, GUI_PREV (gui));
    else
      gui_direct_change_audio_channel (NULL, gui, num_arg);
    break;

  case ACTID_PAUSE:
    gui_pause (NULL, gui, 0, 0);
    break;

  case ACTID_PLAYLIST:
    playlist_main (NULL, gui);
    break;

  case ACTID_TOGGLE_VISIBLITY:
    panel_toggle_visibility (NULL, gui->panel);
    break;

  case ACTID_FLIP_V:
  case ACTID_FLIP_H:
#ifdef XINE_VO_TRANSFORM_FLIP_H
    {
      int flags = gui->transform.flags ^ (id == ACTID_FLIP_H ? 1 : 2);

      flags = xitk_bitmove (flags, 1, XINE_VO_TRANSFORM_FLIP_H)
            | xitk_bitmove (flags, 2, XINE_VO_TRANSFORM_FLIP_V);
      xine_set_param (gui->stream, XINE_PARAM_VO_TRANSFORM, flags);
      flags = xine_get_param (gui->stream, XINE_PARAM_VO_TRANSFORM);
      if (flags != -1) {
        flags = xitk_bitmove (flags, XINE_VO_TRANSFORM_FLIP_H, 1)
              | xitk_bitmove (flags, XINE_VO_TRANSFORM_FLIP_V, 2);
        if (flags != gui->transform.flags) {
          gui->transform.flags = flags;
          osd_message (gui, "%s", gui->transform.msg[flags & 3]);
        }
      }
    }
#endif
    break;

  case ACTID_TOGGLE_FULLSCREEN:
    if ((num_arg != XITK_INT_KEEP) &&
        (!num_arg ^ xitk_bitmove (video_window_mode (gui->vwin, TOGGLE_MODE), FULLSCR_MODE, 1)))
      break;
    gui_set_fullscreen_mode (NULL, gui);
    break;

#ifdef HAVE_XINERAMA
  case ACTID_TOGGLE_XINERAMA_FULLSCREEN:
    if ((num_arg != XITK_INT_KEEP) &&
        (!num_arg ^ xitk_bitmove (video_window_mode (gui->vwin, TOGGLE_MODE), FULLSCR_XI_MODE, 1)))
      break;
    if ((video_window_is_visible (gui->vwin) >= 2) && !gui->use_root_window)
      video_window_mode (gui->vwin, TOGGLE_MODE | FULLSCR_XI_MODE);
    break;
#endif

  case ACTID_TOGGLE_ASPECT_RATIO:
    gui_toggle_aspect (gui, (num_arg != XITK_INT_KEEP) ? num_arg : -1);
    break;

  case ACTID_STREAM_INFOS:
    stream_infos_main (NULL, gui);
    break;

  case ACTID_TOGGLE_INTERLEAVE:
    gui->flags ^= XUI_FLAG_deint;
    osd_message (gui, _("Deinterlace: %s"), (gui->flags & XUI_FLAG_deint) ? _("enabled") : _("disabled"));
    post_deinterlace (gui);
    /* panel_raise_window (gui->panel); */
    break;

  case ACTID_QUIT:
    pthread_mutex_lock (&gui->event_mutex);
    gui->event_pending--;
    if ((gui->event_pending <= 0) && gui->event_reject)
      pthread_cond_signal (&gui->event_safe);
    pthread_mutex_unlock (&gui->event_mutex);
    gui_exit (NULL, gui);
    break;

  case ACTID_PLAY:
    gui_play (NULL, gui);
    break;

  case ACTID_STOP:
    gui_stop (NULL, gui);
    break;

  case ACTID_CLOSE:
    gui_close (NULL, gui);
    break;

  case ACTID_EVENT_SENDER:
    event_sender_main (NULL, gui);
    break;

  case ACTID_SHOW_MRL:
    gui->is_display_mrl = (num_arg != XITK_INT_KEEP) ? num_arg : !gui->is_display_mrl;
    panel_message (gui->panel, NULL);
    playlist_update_playlist (gui);
    break;

  case ACTID_SHOW_TIME:
    gui->runtime_mode = (num_arg != XITK_INT_KEEP) ? num_arg : !gui->runtime_mode;
    panel_update_runtime_display (gui->panel, 0);
    break;

  case ACTID_MRL_NEXT:
    gui_playlist_start_next (gui, (num_arg != XITK_INT_KEEP) ? num_arg : 1);
    break;

  case ACTID_MRL_PRIOR:
    gui_playlist_start_next (gui, (num_arg != XITK_INT_KEEP) ? -num_arg : -1);
    break;

  case ACTID_MRL_SELECT:
    gui_playlist_play (gui, (num_arg != XITK_INT_KEEP) ? num_arg : 0);
    break;

  case ACTID_SETUP:
    setup_main (NULL, gui);
    break;

  case ACTID_EJECT:
    gui_eject (NULL, gui);
    break;

  case ACTID_SET_CURPOS:
    /* Number is a percentage, range [0..100] */
    if (num_arg != XITK_INT_KEEP)
      gui_set_current_position (gui,
        num_arg > 100 ? 65535 : (65535 * num_arg + 50) / 100);
    break;

  case ACTID_SET_CURPOS_0:
  case ACTID_SET_CURPOS_10:
  case ACTID_SET_CURPOS_20:
  case ACTID_SET_CURPOS_30:
  case ACTID_SET_CURPOS_40:
  case ACTID_SET_CURPOS_50:
  case ACTID_SET_CURPOS_60:
  case ACTID_SET_CURPOS_70:
  case ACTID_SET_CURPOS_80:
  case ACTID_SET_CURPOS_90:
  case ACTID_SET_CURPOS_100:
    gui_set_current_position (gui, (id - ACTID_SET_CURPOS_0) * 65535 / 10);
    break;

  case ACTID_SEEK_REL_m:
    if (num_arg != XITK_INT_KEEP)
      gui_seek_relative (gui, -num_arg);
    break;

  case ACTID_SEEK_REL_p:
    if (num_arg != XITK_INT_KEEP)
      gui_seek_relative (gui, num_arg);
    break;

  case ACTID_SEEK_REL_m60:
  case ACTID_SEEK_REL_p60:
  case ACTID_SEEK_REL_m15:
  case ACTID_SEEK_REL_p15:
  case ACTID_SEEK_REL_m30:
  case ACTID_SEEK_REL_p30:
  case ACTID_SEEK_REL_m7:
  case ACTID_SEEK_REL_p7:
    {
      static const int8_t v[] = {-60, 60, -15, 15, -30, 30, -7, 7};
      gui_seek_relative (gui, v[id - ACTID_SEEK_REL_m60]);
    }
    break;

  case ACTID_MRLBROWSER:
    mrl_browser_main (NULL, gui);
    break;

  case ACTID_MUTE:
    panel_toggle_audio_mute (NULL, gui->panel, -1, 0);
    break;

  case ACTID_AV_SYNC_m3600:
  case ACTID_AV_SYNC_RESET:
  case ACTID_AV_SYNC_p3600:
    val = (int)(id - ACTID_AV_SYNC_m3600) * 3600 - 3600;
    if (val)
      val += xine_get_param (gui->stream, XINE_PARAM_AV_OFFSET);
    xine_set_param (gui->stream, XINE_PARAM_AV_OFFSET, val);
    if (gui_playlist_set_av_offs (gui, val))
      mmk_edit_mediamark (gui, NULL, NULL, GUI_MMK_NONE);
    if (val)
      osd_message (gui, _("A/V offset: %s"), pts2str (buf, val));
    else
      osd_message (gui, _("A/V Offset: reset."));
    break;

  case ACTID_SV_SYNC_m:
  case ACTID_SV_SYNC_RESET:
  case ACTID_SV_SYNC_p:
    val = (int)(id - ACTID_SV_SYNC_m) * 3600 - 3600;
    if (val)
      val += xine_get_param (gui->stream, XINE_PARAM_SPU_OFFSET);
    xine_set_param (gui->stream, XINE_PARAM_SPU_OFFSET, val);
    gui_playlist_set_spu_offs (gui, val);
      mmk_edit_mediamark (gui, NULL, NULL, GUI_MMK_NONE);
    if (val)
      osd_message (gui, _("SPU Offset: %s"), pts2str (buf, val));
    else
      osd_message (gui, _("SPU Offset: reset."));
    break;

  case ACTID_SPEED_FAST:
    gui_nextprev_speed (NULL, GUI_NEXT (gui));
    break;

  case ACTID_SPEED_SLOW:
    gui_nextprev_speed (NULL, GUI_PREV (gui));
    break;

  case ACTID_SPEED_RESET:
    gui_nextprev_speed (NULL, GUI_RESET (gui));
    break;

  case ACTID_APP:
    pplugin_main (NULL, &gui->post_audio);
    break;

  case ACTID_APP_ENABLE:
    gui->flags ^= XUI_FLAG_post_a;
    osd_message (gui, _("Audio post plugins: %s."), (gui->flags & XUI_FLAG_post_a) ? _("enabled") : _("disabled"));
    pplugin_update_enable_button (&gui->post_audio);
    if (pplugin_is_post_selected (&gui->post_audio))
      pplugin_rewire_posts (&gui->post_audio);
    break;

  case ACTID_pVOLUME:
    if ((gui->mixer.type_volume == SOUND_CARD_MIXER) && (gui->mixer.level[SOUND_CARD_MIXER] >= 0)) {
      num_arg = (num_arg == XITK_INT_KEEP) ? 1 : num_arg;
      gui_set_audio_vol (gui, GUI_AUDIO_VOL_RELATIVE + num_arg);
      break;
    }
    /* fall through */
  case ACTID_pAMP:
    num_arg = (num_arg == XITK_INT_KEEP) ? 1 : num_arg;
    gui_set_amp_level (gui, GUI_AUDIO_VOL_RELATIVE + num_arg);
    break;

  case ACTID_mVOLUME:
    if ((gui->mixer.type_volume == SOUND_CARD_MIXER) && (gui->mixer.level[SOUND_CARD_MIXER] >= 0)) {
      num_arg = (num_arg == XITK_INT_KEEP) ? 1 : num_arg;
      gui_set_audio_vol (gui, GUI_AUDIO_VOL_RELATIVE - num_arg);
      break;
    }
    /* fall through */
  case ACTID_mAMP:
    num_arg = (num_arg == XITK_INT_KEEP) ? 1 : num_arg;
    gui_set_amp_level (gui, GUI_AUDIO_VOL_RELATIVE - num_arg);
    break;

  case ACTID_AMP_RESET:
    gui_set_amp_level (gui, 100);
    break;

  case ACTID_SNAPSHOT:
    panel_snapshot (NULL, gui->panel);
    break;

  case ACTID_ZOOM_IN:
  case ACTID_ZOOM_OUT:
  case ACTID_ZOOM_X_IN:
  case ACTID_ZOOM_X_OUT:
  case ACTID_ZOOM_Y_IN:
  case ACTID_ZOOM_Y_OUT:
    {
      static const int8_t z[ACTID_ZOOM_RESET + 1 - ACTID_ZOOM_IN][2] = {
        [ACTID_ZOOM_IN    - ACTID_ZOOM_IN] = { 1,  1},
        [ACTID_ZOOM_OUT   - ACTID_ZOOM_IN] = {-1, -1},
        [ACTID_ZOOM_X_IN  - ACTID_ZOOM_IN] = { 1,  0},
        [ACTID_ZOOM_X_OUT - ACTID_ZOOM_IN] = {-1,  0},
        [ACTID_ZOOM_Y_IN  - ACTID_ZOOM_IN] = { 0,  1},
        [ACTID_ZOOM_Y_OUT - ACTID_ZOOM_IN] = { 0, -1}
      };
      int dx = z[id - ACTID_ZOOM_IN][0], dy = z[id - ACTID_ZOOM_IN][1];
      dx += xine_get_param (gui->stream, XINE_PARAM_VO_ZOOM_X),
      dy += xine_get_param (gui->stream, XINE_PARAM_VO_ZOOM_Y);
      xine_set_param (gui->stream, XINE_PARAM_VO_ZOOM_X, dx);
      xine_set_param (gui->stream, XINE_PARAM_VO_ZOOM_Y, dy);
      osd_message (gui, "%s", kbindings_action_name (gui->kbindings, ~id));
    }
    break;

  case ACTID_ZOOM_RESET:
    xine_set_param (gui->stream, XINE_PARAM_VO_ZOOM_X, 100);
    xine_set_param (gui->stream, XINE_PARAM_VO_ZOOM_Y, 100);
    config_update_num (gui->xine, "video.output.horizontal_position", 50);
    config_update_num (gui->xine, "video.output.vertical_position", 50);
    osd_message (gui, "%s", kbindings_action_name (gui->kbindings, ~id));
    break;

  case ACTID_ZOOM_LEFT:
  case ACTID_ZOOM_RIGHT:
  case ACTID_ZOOM_UP:
  case ACTID_ZOOM_DOWN:
    {
      static const char * const key[] = {
        [ACTID_ZOOM_LEFT  - ACTID_ZOOM_LEFT] = "video.output.horizontal_position",
        [ACTID_ZOOM_RIGHT - ACTID_ZOOM_LEFT] = "video.output.horizontal_position",
        [ACTID_ZOOM_UP    - ACTID_ZOOM_LEFT] = "video.output.vertical_position",
        [ACTID_ZOOM_DOWN  - ACTID_ZOOM_LEFT] = "video.output.vertical_position"
      };
      static const int8_t d[] = {
        [ACTID_ZOOM_LEFT  - ACTID_ZOOM_LEFT] = -1,
        [ACTID_ZOOM_RIGHT - ACTID_ZOOM_LEFT] =  1,
        [ACTID_ZOOM_UP    - ACTID_ZOOM_LEFT] = -1,
        [ACTID_ZOOM_DOWN  - ACTID_ZOOM_LEFT] =  1
      };
      xine_cfg_entry_t  cfg_entry;
      if (xine_config_lookup_entry (gui->xine, key[id - ACTID_ZOOM_LEFT], &cfg_entry)) {
        cfg_entry.num_value += d[id - ACTID_ZOOM_LEFT];
        if (cfg_entry.num_value < 0)
          cfg_entry.num_value = 0;
        else if (cfg_entry.num_value > 100)
          cfg_entry.num_value = 100;
        config_update_num (gui->xine, key[id - ACTID_ZOOM_LEFT], cfg_entry.num_value);
        osd_message (gui, "%s", kbindings_action_name (gui->kbindings, ~id));
      }
    }
    break;

  case ACTID_GRAB_POINTER:
    if (!gui->cursor_grabbed) {
      if (panel_is_visible (gui->panel) < 2)
        video_window_grab_pointer (gui->vwin, 1);
      gui->cursor_grabbed = 1;
    } else {
      video_window_grab_pointer (gui->vwin, 0);
      gui->cursor_grabbed = 0;
    }
    break;

  case ACTID_TOGGLE_TVMODE:
    /* Toggle TV Modes on the dxr3 */
    xine_set_param (gui->stream, XINE_PARAM_VO_TVMODE,
      xine_get_param (gui->stream, XINE_PARAM_VO_TVMODE) + 1);
    osd_message (gui, _("TV Mode: %d"), xine_get_param(gui->stream, XINE_PARAM_VO_TVMODE));
    break;

  case ACTID_TVANALOG:
    tvset_main (NULL, gui);
    break;

  case ACTID_VIEWLOG:
    viewlog_main (NULL, gui);
    break;

  case ACTID_KBEDIT:
    kbedit_main (NULL, gui);
    break;

  case ACTID_KBENABLE:
    if (num_arg == XITK_INT_KEEP)
      gui->kbindings_enabled ^= 1;
    else
      gui->kbindings_enabled = (gui->kbindings_enabled & ~1) | (num_arg & 1);
    if (gui->keyedit)
      kbedit_main (XUI_W_ON, gui);
    osd_message (gui, _("Key bindings: %s"), (gui->kbindings_enabled & 1) ? _("enabled") : _("disabled"));
    break;

  case ACTID_DPMSSTANDBY:
    gui->flags ^= XUI_FLAG_save_screen;
    osd_message (gui, _("Monitor: %s"), (gui->flags & XUI_FLAG_save_screen) ? _("disabled") :_("enabled"));
    if (num_arg > 0)
      xitk_usec_sleep (num_arg);
    if (gui->flags & XUI_FLAG_save_screen)
      xitk_system (0, "xset dpms force standby");
    break;

  case ACTID_SCANPLAYLIST:
    playlist_action (gui, PLAYLIST_SCAN);
    break;

  case ACTID_MMKEDITOR:
    playlist_action (gui, PLAYLIST_CURR_EDIT);
    break;

  case ACTID_SUBSELECT:
    gui_select_sub (gui);
    break;

  case ACTID_LOOPMODE:
    if (num_arg == XITK_INT_KEEP)
      gui->playlist.loop++;
    else
      gui->playlist.loop = num_arg;
    if (gui->playlist.loop >= PLAYLIST_LOOP_MODES_NUM)
      gui->playlist.loop = PLAYLIST_LOOP_NO_LOOP;

    {
      static const char loop_names[PLAYLIST_LOOP_MODES_NUM][28] = {
        [PLAYLIST_LOOP_NO_LOOP]   = N_("Playlist: no loop."),
        [PLAYLIST_LOOP_LOOP]      = N_("Playlist: loop."),
        [PLAYLIST_LOOP_REPEAT]    = N_("Playlist: entry repeat."),
        [PLAYLIST_LOOP_SHUFFLE]   = N_("Playlist: shuffle."),
        [PLAYLIST_LOOP_SHUF_PLUS] = N_("Playlist: shuffle forever.")
      };
      osd_message (gui, "%s", gettext (loop_names[gui->playlist.loop]));
    }
    break;

  case ACTID_ADDMEDIAMARK:
    if (!gui->logo_mode && (xine_get_status (gui->stream) == XINE_STATUS_PLAY)) {
      int v[3];

      if (gui_xine_get_pos_length (gui, gui->stream, v)) {
        v[1] /= 1000;
        gui_playlist_append (gui, gui->mmk.mrl, gui->mmk.ident, gui->mmk.sub, v[1], -1, gui->mmk.av_offset, gui->mmk.spu_offset);
        playlist_update_playlist (gui);
        oxine_playlist_update (gui->oxine);
      }
    }

    break;

#ifdef HAVE_TAR
  case ACTID_SKINDOWNLOAD:
    skin_download (gui, gui->skin_server_url);
    break;
#endif

  case ACTID_OSD_SINFOS:
    osd_stream_infos (gui);
    break;

  case ACTID_OSD_WTEXT:
    osd_message (gui, "%s", str_arg ? str_arg : _("No text to display!"));
    break;

  case ACTID_OSD_MENU:
    oxine_menu (gui->oxine);
    break;

  case ACTID_FILESELECTOR:
    gui_file_selector (gui);
    break;

  case ACTID_HUECONTROLp:
  case ACTID_HUECONTROLm:
  case ACTID_SATURATIONCONTROLp:
  case ACTID_SATURATIONCONTROLm:
  case ACTID_BRIGHTNESSCONTROLp:
  case ACTID_BRIGHTNESSCONTROLm:
  case ACTID_CONTRASTCONTROLp:
  case ACTID_CONTRASTCONTROLm:
  case ACTID_GAMMACONTROLp:
  case ACTID_GAMMACONTROLm:
  case ACTID_SHARPNESSCONTROLp:
  case ACTID_SHARPNESSCONTROLm:
  case ACTID_NOISEREDUCTIONCONTROLp:
  case ACTID_NOISEREDUCTIONCONTROLm:
    control_action (gui->vctrl, id);
    break;

  case ACTID_VPP:
    pplugin_main (NULL, &gui->post_video);
    break;

  case ACTID_VPP_ENABLE:
    gui->flags ^= XUI_FLAG_post_v;
    osd_message (gui, _("Video post plugins: %s."), (gui->flags & XUI_FLAG_post_v) ? _("enabled") : _("disabled"));
    pplugin_update_enable_button (&gui->post_video);
    if (pplugin_is_post_selected (&gui->post_video))
      pplugin_rewire_posts (&gui->post_video);
    break;

  case ACTID_HELP_SHOW:
    help_main (NULL, gui);
    break;

  case ACTID_PLAYLIST_STOP:
    {
      uint32_t mode;
        gui_playlist_lock (gui);
        mode = gui->playlist.control;
        if (num_arg == XITK_INT_KEEP) {
          mode ^= PLAYLIST_CONTROL_STOP;
          mode &= ~PLAYLIST_CONTROL_STOP_PERSIST;
        } else {
          if (num_arg == 0)
            mode &= ~PLAYLIST_CONTROL_STOP;
          else
            mode |= PLAYLIST_CONTROL_STOP;
          mode |= PLAYLIST_CONTROL_STOP_PERSIST;
        }
        gui->playlist.control = mode;
        gui_playlist_unlock (gui);
        osd_message (gui, _("Playlist: %s"),
          (mode & PLAYLIST_CONTROL_STOP)
          ? ((mode & PLAYLIST_CONTROL_STOP_PERSIST) ? _("Stop (persistent)") : _("Stop"))
          : _("Continue"));
      }
    break;

  case ACTID_PVR_SETINPUT:

    /* Parameter (integer, required): input */
    if (num_arg != XITK_INT_KEEP) {
      xine_event_t         xine_event;
      xine_set_v4l2_data_t ev_data;

      /* set input */
      ev_data.input = num_arg;

      /* do not change tuning */
      ev_data.channel = -1;
      ev_data.frequency = -1;

      /* do not set session id */
      ev_data.session_id = -1;

      /* send event */
      xine_event.type = XINE_EVENT_SET_V4L2;
      xine_event.data_length = sizeof(xine_set_v4l2_data_t);
      xine_event.data = &ev_data;
      xine_event.stream = gui->stream;
      xine_event_send(gui->stream, &xine_event);
    }
    break;

  case ACTID_PVR_SETFREQUENCY:

    /* Parameter (integer, required): frequency */
    if (num_arg != XITK_INT_KEEP) {
      xine_event_t         xine_event;
      xine_set_v4l2_data_t ev_data;

      /* do not change input */
      ev_data.input = -1;

      /* change tuning */
      ev_data.channel = -1;
      ev_data.frequency = num_arg;

      /* do not set session id */
      ev_data.session_id = -1;

      /* send event */
      xine_event.type = XINE_EVENT_SET_V4L2;
      xine_event.data_length = sizeof(xine_set_v4l2_data_t);
      xine_event.data = &ev_data;
      xine_event.stream = gui->stream;
      xine_event_send(gui->stream, &xine_event);
    }
    break;

  case ACTID_PVR_SETMARK:

    /* Parameter (integer, required): session_id

       Mark the start of a section in the pvr stream for later saving
       The 'id' can be used to set different marks in the stream. If
       an existing mark is provided, then the mark for that section
       is moved to the current position.
    */
    if (num_arg != XITK_INT_KEEP) {
      xine_event_t         xine_event;
      xine_set_v4l2_data_t ev_data;

      /* do not change input */
      ev_data.input = -1;

      /* do not change tuning */
      ev_data.channel = -1;
      ev_data.frequency = -1;

      /* set session id */
      ev_data.session_id = num_arg;

      /* send event */
      xine_event.type = XINE_EVENT_SET_V4L2;
      xine_event.data_length = sizeof(xine_set_v4l2_data_t);
      xine_event.data = &ev_data;
      xine_event.stream = gui->stream;
      xine_event_send(gui->stream, &xine_event);
    }
    break;

  case ACTID_PVR_SETNAME:

    /* Parameter (string, required): name

       Set the name of the current section in the pvr stream. The name
       is used when the section is saved permanently.
    */
    if (str_arg) {
      xine_event_t         xine_event;
      xine_pvr_save_data_t ev_data;

      /* only set name */
      ev_data.mode = -1;
      ev_data.id = -1;

      /* name of the show, max 255 is hardcoded in xine.h */
      strlcpy (ev_data.name, str_arg, sizeof(ev_data.name));

      /* send event */
      xine_event.type = XINE_EVENT_PVR_SAVE;
      xine_event.data = &ev_data;
      xine_event.data_length = sizeof(xine_pvr_save_data_t);
      xine_event.stream = gui->stream;
      xine_event_send(gui->stream, &xine_event);
    }
    break;

  case ACTID_PVR_SAVE:

    /* Parameter (integer, required): mode

       Save the pvr stream with mode:
         mode = 0 : save from now on
         mode = 1 : save from last mark
         mode = 2 : save entire stream
    */
    if (num_arg != XITK_INT_KEEP) {
      xine_event_t         xine_event;
      xine_pvr_save_data_t ev_data;
      int                  mode = num_arg;

      /* set mode [0..2] */
      if(mode < 0)
	mode = 0;
      if(mode > 2)
	mode = 2;
      ev_data.mode = mode;

      /* do not set id and name */
      ev_data.id = 0;
      ev_data.name[0] = '\0';

      /* send event */
      xine_event.type = XINE_EVENT_PVR_SAVE;
      xine_event.data = &ev_data;
      xine_event.data_length = sizeof(xine_pvr_save_data_t);
      xine_event.stream = gui->stream;
      xine_event_send(gui->stream, &xine_event);
    }
    break;

  case ACTID_PLAYLIST_OPEN:
    if (str_arg) {
        gui_playlist_add_item (gui, str_arg, 1, GUI_ITEM_TYPE_PLAYLIST, 1);
        gui_current_set_index (gui, GUI_MMK_CURRENT);
        playlist_update_playlist (gui);
        oxine_playlist_update (gui->oxine);
        if (gui->playlist.num > 0)
          panel_playback_ctrl (gui->panel, 1);
    } else {
        playlist_action (gui, PLAYLIST_LOAD);
    }
    break;

  default:
    break;
  }

  pthread_mutex_lock (&gui->event_mutex);
  gui->event_pending--;
  if ((gui->event_pending <= 0) && gui->event_reject)
    pthread_cond_signal (&gui->event_safe);
  pthread_mutex_unlock (&gui->event_mutex);
}

/*
 * top-level event handler
 */
int gui_handle_be_event (void *data, const xitk_be_event_t *e) {
  gGui_t *gui = data;

  if (!gui || !e)
    return 0;
  switch (e->type) {
    case XITK_EV_KEY_UP:
      if (gui->flags & XUI_FLAG_wait_key_up) {
        gui->flags &= ~XUI_FLAG_wait_key_up;
        gui_action_args (gui, ACTID_DPMSSTANDBY, 900000, NULL);
        break;
      }
      return 0;
    case XITK_EV_KEY_DOWN:
      if ((e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_MENU)) {
        video_window_menu (gui, e->wl, e->w, e->h);
        return 1;
      }
      /* fall through */
    case XITK_EV_BUTTON_UP:
      {
        action_id_t a;
        if (gui->stdctl_enable)
          stdctl_keypress (gui, e->utf8);
        a = kbindings_aid_from_be_event (gui->kbindings, e, gui);
        if (!a)
          return 0;
        {
          struct timespec ts;
          int d;
          xitk_gettime_ts (&ts);
          /* system time minus server time, hopefulle wrap safe. */
          d = (ts.tv_sec % 1000000) * 1000 + ts.tv_nsec / 1000000 - e->time % 1000000000;
          while (d < 0)
            d += 1000000000;
          /* 1/4s later than last time is suspicious. cap at 1 minute for wrap safety. */
          if (XITK_0_TO_MAX_MINUS_1 (d - 256 - gui->be_brake.diff, 60000 + 256) && (gui->be_brake.n > 0)) {
            gui->be_brake.n++;
            if (gui->verbosity >= 3)
              printf ("gui.action.brake: dropped event #%d @ diff %d.\n", (int)a, d);
            return 0;
          }
          if ((gui->be_brake.n > 1) && (gui->verbosity >= 2))
            printf ("gui.action.brake: dropped %d late events @ diff %d.\n", gui->be_brake.n - 1, d);
          gui->be_brake.diff = d;
          gui->be_brake.n = 1;
        }
        if ((a == ACTID_DPMSSTANDBY) && (e->type == XITK_EV_KEY_DOWN)) {
          /* the following key up would wake up the monitor again, so defer this one. */
          gui->flags |= XUI_FLAG_wait_key_up;
          /* we dont see the modifier key events here, so wait some fixed time if any of them are pressed. */
          /* if (e->qual & (MODIFIER_SHIFT | MODIFIER_CTRL | MODIFIER_META | MODIFIER_MOD4 | MODIFIER_MOD5))
            xitk_usec_sleep (900000); */
        } else {
          gui_action_id (gui, a);
        }
      }
      break;
    case XITK_EV_BUTTON_DOWN:
      if (e->code == 3) {
        video_window_menu (gui, e->wl, e->w, e->h);
        return 1;
      }
      break;
    case XITK_EV_DND:
      gui_dndcallback (gui, e->utf8);
      break;
    default: return 0;
  }
  return 1;
}

/** call with playlist locked. return 0 (OK), 1 (fail), 2 (all played). */
static int _gui_playlist_play_current (gGui_t *gui, int post_update) {
  if (!post_update ||
    (gui->playlist.mmk && gui->playlist.mmk[gui->playlist.cur] && gui->playlist.mmk[gui->playlist.cur]->mrl)) {
    mediamark_t *mmk, temp = {.start = 0};
    int changed_flags = 0;

    if (post_update) {
      mmk = &gui->mmk;
      changed_flags = mediamark_copy (&mmk, gui->playlist.mmk[gui->playlist.cur]) | MMK_CHANGED_INDEX;
      gui->playlist.ref_append = gui->playlist.cur;
    }
    gui->mmk.played |= 3;
    if ((gui->playlist.cur >= 0) && gui->playlist.mmk && gui->playlist.mmk[gui->playlist.cur]
      && (gui->mmk.mrl == gui->playlist.mmk[gui->playlist.cur]->mrl))
      gui_playlist_mark_played (gui);
    mmk = &temp;
    mediamark_copy (&mmk, &gui->mmk);
    gui_playlist_unlock (gui);

    if (post_update)
      gui_pl_updated (gui, changed_flags);

    if (gui_xine_open_and_play (gui, temp.mrl, temp.sub, 0, temp.start, temp.av_offset, temp.spu_offset, !temp.alt[0])) {
      gui_playlist_lock (gui);
      mediamark_clear (&temp);
      return 0;
    }
    if (temp.alt[0]) {
      const char *alt;
      for (alt = mediamark_get_alternate_mrl (&temp, 3); alt; alt = mediamark_get_alternate_mrl (&temp, 0)) {
        if (gui_xine_open_and_play (gui, alt, temp.sub, 0, temp.start, temp.av_offset, temp.spu_offset, 1)) {
          gui_playlist_lock (gui);
          mediamark_clear (&temp);
          return 0;
        }
      }
    }
    gui_playlist_lock (gui);
    mediamark_clear (&temp);
  }
  return gui_playlist_all_played (gui) ? 2 : 1;
}

void gui_play (xitk_widget_t *w, void *data) {
  gGui_t *gui = data;
  int status;

  (void)w;
  status = xine_get_status (gui->stream);

  gui_playlist_lock (gui);

  if (status != XINE_STATUS_PLAY) {
    if (!gui->playlist.num) {
      gui_playlist_unlock (gui);
      video_window_reset_ssaver (gui->vwin, 0);
      return;
    }
  } else {
    if (gui->logo_mode) {
      gui_playlist_unlock (gui);
      gui->ignore_next = 1;
      xine_stop (gui->stream);
      status = xine_get_status (gui->stream);
      gui->ignore_next = 0;
      gui_playlist_lock (gui);
    }
  }

  if (status != XINE_STATUS_PLAY) {
    int ret;

    if (!strncmp (gui->mmk.ident, "xine-ui version", 15)) {
      gui_playlist_unlock (gui);
      video_window_reset_ssaver (gui->vwin, 0);
      gui_msg (gui, XUI_MSG_ERROR, _("No MRL (input stream) specified"));
      return;
    }

    ret = _gui_playlist_play_current (gui, 0);
    gui_playlist_unlock (gui);
    if (!ret) {
      panel_check_pause (gui->panel);
      video_window_reset_ssaver (gui->vwin, 1);
      return;
    }
    if ((ret == 2) && (gui->flags & XUI_FLAG_start_quit)) {
      gui_exit (NULL, gui);
      return;
    }
    gui_display_logo (gui);

  } else {
    int oldspeed;
    gui_playlist_unlock (gui);
    oldspeed = xine_get_param (gui->stream, XINE_PARAM_SPEED);
    xine_set_param (gui->stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
    video_window_reset_ssaver (gui->vwin, 1);
    if (oldspeed != XINE_SPEED_NORMAL)
      osd_play_status (gui, 0);
  }

  panel_check_pause (gui->panel);
}

int gui_playlist_play_current (gGui_t *gui) {
  int ret;

  gui_playlist_lock (gui);
  ret = _gui_playlist_play_current (gui, 0);
  gui_playlist_unlock (gui);
  video_window_reset_ssaver (gui->vwin, !ret);
  return ret;
}

/*
 * Start playback of an entry in playlist
 */
int gui_playlist_play (gGui_t *gui, int idx) {
  int ret = 1;

  osd_hide (gui);
  panel_reset_slider (gui->panel);

  if (!(gui_current_set_index (gui, idx) & MMK_CHANGED_FAIL)) {
    gui_playlist_lock (gui);
    ret = _gui_playlist_play_current (gui, 0);
    gui_playlist_unlock (gui);
    video_window_reset_ssaver (gui->vwin, !ret);
  }
  return ret;
}

/*
 * Start playback to next entry in playlist (or stop the engine, then display logo).
 */
void gui_playlist_start_next (gGui_t *gui, int by) {
  int again;

  if (!gui)
    return;
  if (gui->ignore_next)
    return;

  osd_hide (gui);

  if (gui->skip_by_chapter && by && xine_get_stream_info (gui->stream, XINE_STREAM_INFO_HAS_CHAPTERS)) {
    if (by < 0) {
      while (by++)
        gui_action_id (gui, ACTID_EVENT_PRIOR);
    } else {
      while (by--)
        gui_action_id (gui, ACTID_EVENT_NEXT);
    }
    panel_check_pause (gui->panel);
    return;
  }

  panel_reset_slider (gui->panel);

  gui_playlist_lock (gui);

  if (gui->playlist.control & PLAYLIST_CONTROL_STOP) {
    if (!(gui->playlist.control & PLAYLIST_CONTROL_STOP_PERSIST))
      gui->playlist.control &= ~PLAYLIST_CONTROL_STOP;
    gui_playlist_unlock (gui);
    gui_display_logo (gui);
    return;
  }

  if (!panel_playback_ctrl (gui->panel, gui->playlist.num > 0)) {
    gui_current_set_index (gui, GUI_MMK_NONE);
    gui_playlist_unlock (gui);
    gui_display_logo (gui);
    return;
  }

  /* playback fail loop, try each entry at most 2 times. */
  again = 2;
  do {
    if ((gui->playlist.loop == PLAYLIST_LOOP_SHUFFLE) || (gui->playlist.loop == PLAYLIST_LOOP_SHUF_PLUS)) {
      if (gui_playlist_all_played (gui)) {
        mediamark_reset_played_state (gui);
        if (gui->playlist.loop == PLAYLIST_LOOP_SHUFFLE) {
          if (by) {
            gui_playlist_unlock (gui);
            return;
          }
          break;
        }
        again--;
      }
      gui->playlist.cur = mediamark_get_shuffle_next (gui);
    } else if (gui->playlist.loop == PLAYLIST_LOOP_REPEAT) {
      again--;
    } else { /* PLAYLIST_LOOP_NO_LOOP || PLAYLIST_LOOP_LOOP */
      int newcur = gui->playlist.cur + by + (by == 0);
      if (!XITK_0_TO_MAX_MINUS_1 (newcur, gui->playlist.num)) {
        if (gui->playlist.loop == PLAYLIST_LOOP_NO_LOOP) {
          if (by) {
            gui_playlist_unlock (gui);
            return;
          }
          if (gui_playlist_all_played (gui)) {
            mediamark_reset_played_state (gui);
            break;
          }
          gui_playlist_unlock (gui);
          gui_display_logo (gui);
          return;
        }
        again--;
        newcur = (newcur + gui->playlist.num) % gui->playlist.num;
      }
      gui->playlist.cur = newcur;
    }
    /* try playing */
    if (!_gui_playlist_play_current (gui, 1)) {
      gui_playlist_unlock (gui);
      video_window_reset_ssaver (gui->vwin, 1);
      panel_check_pause (gui->panel);
      return;
    }
  } while (again);
  /* bail out */
  gui_playlist_unlock (gui);
  if (gui->flags & XUI_FLAG_start_quit)
    gui_exit (NULL, gui);
  else
    gui_display_logo (gui);
}

static void *gui_probe_thread (void *data) {
  gGui_t *gui = data;
  mediamark_t temp = {.start = 0}, *mmk = &temp;
  xine_stream_t *stream;
  int index = 0, probed = 0;

  typedef union {
    char s[4];
    uint32_t v;
  } _s4_t;
  const _s4_t _file = {.s = "file"};

  pthread_detach (pthread_self ());

  stream = xine_stream_new (gui->xine, NULL, NULL);
  if (!stream)
    return NULL;

  if (gui->verbosity >= 2)
    printf ("gui.playlist.probe.start ().\n");

  gui_playlist_lock (gui);
  for (index = 0; (gui->playlist.probe_mode & 4); index++) {
    if (gui->playlist.probe_gen != gui->playlist.gen)
      gui->playlist.probe_gen = gui->playlist.gen, index = 0;
    if (index >= gui->playlist.num)
      break;
    for (; index < gui->playlist.num; index++) {
      /* already played or probed? */
      if (gui->playlist.mmk[index]->played & 2)
        continue;
      /* bogus entry? */
      if (!gui->playlist.mmk[index]->mrl)
        continue;
      /* does user want probing this type? */
      if ((gui->playlist.probe_mode & 3) != 3) {
        _s4_t prot[8];
        char *e;
        uint32_t type = 1;
        strlcpy (prot[0].s, gui->playlist.mmk[index]->mrl, sizeof (prot));
        e = strstr (prot[0].s, ":/");
        if (e)
          type = ((e == prot[0].s + 4) && ((prot[0].v | 0x20202020) == _file.v)) ? 1 : 2;
        if (!(gui->playlist.probe_mode & type))
          continue;
      }
      break;
    }
    if (index >= gui->playlist.num)
      break;
    gui->playlist.mmk[index]->played |= 2;
    gui->playlist.probed++;
    /* copy entry */
    mediamark_copy (&mmk, gui->playlist.mmk[index]);
    gui_playlist_unlock (gui);
    /* probe */
    if (gui->verbosity >= 2)
      printf ("gui.playlist.probe (%d, \"%s\").\n", index, mmk->mrl);
    {
      char ident[2048];
      int length = -1, n = (mmk->ident && strstr (mmk->mrl, mmk->ident)) ? 1 : 0;

      if (xine_open (stream, mmk->mrl)) {
        if (xine_get_pos_length (stream, NULL, NULL, &length) && (length != -1))
          mmk->length = length;
        if (!n && stream_infos_get_ident_from_stream (stream, ident, sizeof (ident)))
          mediamark_set_str_val (&mmk, ident, MMK_VAL_IDENT);
        xine_close (stream);
      }
    }
    probed++;

    gui_playlist_lock (gui);
    if (gui->playlist.probe_gen != gui->playlist.gen)
      continue;
    /* copy back */
    if (mediamark_copy (gui->playlist.mmk + index, mmk)) {
      gui_playlist_now_changed (gui);
      gui->playlist.probe_gen = gui->playlist.gen;
    }
  }
  mediamark_clear (mmk);
  gui->playlist.probe_mode &= ~4;
  gui_playlist_unlock (gui);

  if (probed) {
    xitk_lock (gui->xitk, 1);
    playlist_update_playlist (gui);
    /* panel_message (gui->panel, NULL); */
    xitk_lock (gui->xitk, 0);
  }

  xine_dispose (stream);

  if (gui->verbosity >= 2)
    printf ("playlist.probe.done (%d).\n", probed);

  return NULL;
}

static void _gui_playlist_probe_stop (gGui_t *gui) {
  gui_playlist_lock (gui);
  gui->playlist.probe_mode &= ~4;
  gui_playlist_unlock (gui);
}

void gui_playlist_probe_start (gGui_t *gui) {
  gui_playlist_lock (gui);
  if ((gui->playlist.probe_gen != gui->playlist.gen) && (gui->playlist.probe_mode & 3)
    && !(gui->playlist.probe_mode & 4)) {
    pthread_t pth;
    int err;

    gui->playlist.probe_gen = gui->playlist.gen;
    gui->playlist.probe_mode |= 4;
    gui_playlist_unlock (gui);
    err = pthread_create (&pth, NULL, gui_probe_thread, gui);
    if (err) {
      if (gui->verbosity >= 1)
        printf (_("%s(): can't create new thread (%s)\n"), __XINE_FUNCTION__, strerror (err));
      gui_playlist_lock (gui);
      gui->playlist.probe_mode &= ~4;
      gui_playlist_unlock (gui);
    }
    return;
  }
  gui_playlist_unlock (gui);
}

static void _gui_playlist_probe_files (void *data, xine_cfg_entry_t *entry) {
  gGui_t *gui = data;
  uint32_t newf = entry->num_value ? 1 : 0;

  gui_playlist_lock (gui);
  if ((gui->playlist.probe_mode & 1) != newf) {
    gui->playlist.probe_mode = (gui->playlist.probe_mode & ~1) | newf;
    gui->playlist.probe_gen = 0;
    if (newf)
      gui_playlist_probe_start (gui);
  }
  gui_playlist_unlock (gui);
}

static void _gui_playlist_probe_network (void *data, xine_cfg_entry_t *entry) {
  gGui_t *gui = data;
  uint32_t newf = entry->num_value ? 2 : 0;

  gui_playlist_lock (gui);
  if ((gui->playlist.probe_mode & 2) != newf) {
    gui->playlist.probe_mode = (gui->playlist.probe_mode & ~2) | newf;
    gui->playlist.probe_gen = 0;
    if (newf)
      gui_playlist_probe_start (gui);
  }
  gui_playlist_unlock (gui);
}

static void _gui_playlist_probe_init (gGui_t *gui) {
  gui->playlist.probe_gen = 0;
  gui->playlist.probe_mode =
    (xine_config_register_bool (gui->xine, "gui.playlist.probe_files", 1,
        _("Early get meta info from file items in playlists."),
        _("Get title, length etc. from not yet played files in list."),
        CONFIG_LEVEL_ADV, _gui_playlist_probe_files, gui) & 1) |
    ((xine_config_register_bool (gui->xine, "gui.playlist.probe_network", 0,
        _("Early get meta info from remote items in playlists."),
        _("Get title, length etc. from not yet played network streams in list.\n"
          "This may be slow and hazardous, and is thus not recommended."),
        CONFIG_LEVEL_ADV, _gui_playlist_probe_network, gui) & 1) << 1);
}

void gui_deinit (gGui_t *gui) {
#ifdef HAVE_XINE_CONFIG_UNREGISTER_CALLBACKS
  xine_config_unregister_callbacks (gui->xine, NULL, NULL, gui, sizeof (*gui));
#endif
  _gui_playlist_probe_stop (gui);
  pthread_mutex_lock (&gui->event_mutex);
  gui->event_reject = 1;
  while (gui->event_pending > 0)
    pthread_cond_wait (&gui->event_safe, &gui->event_mutex);
  pthread_mutex_unlock (&gui->event_mutex);
  skin_deinit (gui);
}

static void splash_create (gGui_t *gui, const char *skin_path) {
  char buf[2048];
  xitk_image_t *xim;
  const char *name = NULL;

  if (!(gui->flags & XUI_FLAG_splash))
    return;
  if (xitk_filetype (skin_path) == XITK_FILETYPE_DIR) {
    char *p = buf, *e = buf + sizeof (buf) - 32;

    p += strlcpy (p, skin_path, e - p);
    if (p > e)
      p = e;
    memcpy (p, "/xine_splash.png", 17); p += 13;
    if (xitk_filetype (buf) == XITK_FILETYPE_FILE)
      name = buf;
    if (!name) {
      memcpy (p, "jpg", 4);
      if (xitk_filetype (buf) == XITK_FILETYPE_FILE)
        name = buf;
    }
  }
  if (!name)
    name = XINE_SPLASH;

  if ((xim = xitk_image_new (gui->xitk, name, 0, 0, 0))) {
    int w = xitk_image_width (xim), h = xitk_image_height (xim);

    gui->splash_win = xitk_window_create_window_ext (gui->xitk, -1, -1, w, h,
        _("xine Splash"), NULL, "xine", 0, 1, gui->icon, xim);
    if (gui->splash_win) {
      xitk_window_set_wm_window_type (gui->splash_win, WINDOW_TYPE_SPLASH);
      xitk_window_flags (gui->splash_win, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
      xitk_window_raise_window (gui->splash_win);
    }
    xitk_image_free_image (&xim);
  }
}

static void splash_destroy (gGui_t *gui) {
  if (gui->splash_win) {
    xitk_lock (gui->xitk, 1);
    xitk_window_flags (gui->splash_win, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, 0);
    xitk_window_destroy_window (gui->splash_win);
    gui->splash_win = NULL;
    xitk_lock (gui->xitk, 0);
  }
}
/*
 * Initialize the GUI
 */

void gui_init (gGui_t *gui, gui_init_params_t *p) {
  int    i;
  pthread_mutexattr_t attr;
  int    use_x_lock_display;
  int    use_synchronized_x11;

  pthread_mutexattr_init(&attr);
  pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);
  pthread_mutex_init (&gui->event_mutex, &attr);
  pthread_cond_init (&gui->event_safe, NULL);

  gui->event_reject = 1;
  gui->event_pending = 0;

  gui->action_num_arg = XITK_INT_KEEP;

  /*
   * init playlist
   */
  for (i = 0; i < p->num_files; i++) {
    const char *file = p->filenames[i];

    gui_playlist_add_item (gui, file, GUI_MAX_DIR_LEVELS, GUI_ITEM_TYPE_AUTO, 0);
  }

  gui->playlist.cur = gui->playlist.num ? 0 : -1;

  if((gui->playlist.loop == PLAYLIST_LOOP_SHUFFLE) ||
     (gui->playlist.loop == PLAYLIST_LOOP_SHUF_PLUS))
    gui->playlist.cur = mediamark_get_shuffle_next (gui);

  gui->is_display_mrl = 0;
  /* gui->new_pos        = -1; */

  /*
   *
   */

  gui->transform.msg[0] = _("Video upright");
  gui->transform.msg[1] = _("Video mirrored");
  gui->transform.msg[2] = _("Video upside down");
  gui->transform.msg[3] = _("Video upside down and mirrored");
  gui->transform.flags = 0;

  use_x_lock_display =
    xine_config_register_bool (gui->xine, "gui.use_XLockDisplay", 1,
        _("Enable extra XLib thread safety."),
        _("This is needed for some very old XLib/XCB versions.\n"
          "Otherwise, it may boost or brake performance - just try out."),
        CONFIG_LEVEL_ADV, NULL, NULL);

  use_synchronized_x11 = xine_config_register_bool (gui->xine, "gui.xsynchronize", 0,
    _("Synchronized X protocol (debug)"),
    _("Do X transactions in synchronous mode. Very slow, use only for debugging!"),
    CONFIG_LEVEL_ADV, CONFIG_NO_CB, CONFIG_NO_DATA);

  gui->layer_above = xine_config_register_bool (gui->xine, "gui.layer_above", 0,
    _("Windows stacking"),
    _("Use WM layer property to place windows on top\n"
      "except for the video window in non-fullscreen mode."),
    CONFIG_LEVEL_ADV, layer_above_cb, gui) ? 1 : 0;

  gui->layer_above |= xine_config_register_bool (gui->xine, "gui.always_layer_above", 0,
    _("Windows stacking (more)"),
    _("Use WM layer property to place windows on top\n"
      "for all windows (surpasses the 'layer_above' setting)."),
    CONFIG_LEVEL_ADV, always_layer_above_cb, gui) ? 2 : 0;

  gui->snapshot_location = xine_config_register_filename (gui->xine, "gui.snapshotdir",
    xine_get_homedir(),
    XINE_CONFIG_STRING_IS_DIRECTORY_NAME,
    _("Snapshot location"),
    _("Where snapshots will be saved."),
    CONFIG_LEVEL_BEG, snapshot_loc_cb, gui);

  gui->ssaver_timeout = xine_config_register_num (gui->xine, "gui.screensaver_timeout", 10,
    _("Screensaver reset interval (s)"),
    _("Time, in seconds, between two faked events to keep a screensaver quiet, 0 to disable."),
    CONFIG_LEVEL_ADV, ssaver_timeout_cb, gui);

  gui->skip_by_chapter = xine_config_register_bool (gui->xine, "gui.skip_by_chapter", 1,
    _("Chapter hopping"),
    _("Play next|previous chapter instead of mrl (dvdnav)"),
    CONFIG_LEVEL_ADV, skip_by_chapter_cb, gui);

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.auto_video_output_visibility", 0,
    _("Visibility behavior of output window"),
    _("Hide video output window if there is no video in the stream"),
    CONFIG_LEVEL_ADV, auto_vo_visibility_cb, gui), 1, XUI_FLAG_auto_vo_vis);

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.auto_panel_visibility", 0,
    _("Visiblility behavior of panel"),
    _("Automatically show/hide panel window, according to auto_video_output_visibility"),
    CONFIG_LEVEL_ADV, auto_panel_visibility_cb, gui), 1, XUI_FLAG_auto_panel_vis);

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.eventer_sticky", 1,
    _("Event sender behavior"),
    _("Event sender window stick to main panel"),
    CONFIG_LEVEL_ADV, event_sender_sticky_cb, gui), 1, XUI_FLAG_eventer_sticky);

  gui->visual_anim.enabled = xine_config_register_enum (gui->xine, "gui.visual_anim", 1, /* Post plugin */
    (char**)visual_anim_style,
    _("Visual animation style"),
    _("Display some video animations when current stream is audio only (eg: mp3)."),
    CONFIG_LEVEL_BEG, visual_anim_cb, gui);

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.sinfo_auto_update", 0,
    _("Stream information"),
    _("Update stream information (in stream info window) each half seconds."),
    CONFIG_LEVEL_ADV, stream_info_auto_update_cb, gui), 1, XUI_FLAG_sinfo_update);

  {
    const char *server = xine_config_register_string (gui->xine, "gui.skin_server_url", SKIN_SERVER_URL,
      _("Skin Server Url"),
      _("From where we can get skins."),
      CONFIG_LEVEL_ADV, skin_server_url_cb, gui);
    if (!gui->skin_server_url)
      gui->skin_server_url = strdup (server);
  }

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.smart_mode", 1,
    _("Change xine's behavior for unexperienced user"),
    _("In this mode, xine take some decisions to simplify user's life."),
    CONFIG_LEVEL_BEG, smart_mode_cb, gui), 1, XUI_FLAG_smart_mode);

  gui->flags |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.play_anyway", 0,
    _("Ask user for playback with unsupported codec"),
    _("If xine don't support audio or video codec of current stream "
      "the user will be asked if the stream should be played or not"),
    CONFIG_LEVEL_BEG, play_anyway_cb, gui), 1, XUI_FLAG_play_anyway);

  gui->experience_level = (xine_config_register_enum (gui->xine, "gui.experience_level", 0,
    (char**)exp_levels,
    _("Configuration experience level"),
    _("Level of user's experience, this will show more or less configuration options."),
    CONFIG_LEVEL_BEG, exp_level_cb, gui)) * 10;

  gui->mixer.level[SOFTWARE_MIXER] = xine_config_register_range (gui->xine, "gui.amp_level",
    100, 0, 200,
    _("Amplification level"),
    NULL,
    CONFIG_LEVEL_DEB, dummy_config_cb, gui);

  if (gui->flags & XUI_FLAG_splash)
    gui->flags ^= xitk_bitmove (~xine_config_register_bool (gui->xine, "gui.splash", 1,
      _("Display splash screen"),
      _("If enabled, xine will display its splash screen"),
      CONFIG_LEVEL_BEG, dummy_config_cb, gui), 1, XUI_FLAG_splash);

  {
    int v = xine_config_register_enum (gui->xine, "gui.audio_mixer_method",
        SOUND_CARD_MIXER, (char**)mixer_control_method,
        _("Audio mixer control method"),
        _("Which method used to control audio volume."),
        CONFIG_LEVEL_ADV, audio_mixer_method_cb, gui);
    xui_mixer_type_t type = v;
    if (type >= LAST_MIXER)
      type = SOFTWARE_MIXER;
    gui->mixer.type_volume = gui->mixer.type_mute = type;
  }

  gui->shortcut_style = xine_config_register_enum (gui->xine, "gui.shortcut_style", 0,
    (char**)shortcut_style,
    _("Menu shortcut style"),
    _("Shortcut representation in menu, 'Ctrl,Alt' or 'C,M'."),
    CONFIG_LEVEL_ADV, shortcut_style_cb, gui);

  _gui_playlist_probe_init (gui);

  /*
   * create and map panel and video window
   */

  xine_pid = getppid();

  gui->xitk = xitk_init (p->prefered_visual, p->install_colormap,
                         use_x_lock_display, use_synchronized_x11,
                         gui->verbosity);

  if (!gui->xitk) {
    printf ("gui.init: ERROR: xiTK engine unavailable.\n");
    exit (1);
  }

  /*
   * create an icon pixmap
   */
  if (!(gui->icon = xitk_image_new (gui->xitk, XINE_SKINDIR "/xine_64.png", 0, 0, 0)))
    gui->icon = xitk_image_new (gui->xitk, (const char *)icon_datas, -1, 40, 40);

  splash_create (gui, skin_init (gui));
  skin_load (gui);

  gui->on_quit = 0;
  gui->running = 1;

  gui->vwin = video_window_init (gui, p->window_id, p->borderless, p->geometry,
    xitk_bitmove (gui->flags, XUI_FLAG_start_vo_vis, 1), p->prefered_visual, p->install_colormap, use_x_lock_display);
  if (!gui->vwin)
    printf ("gui.init: ERROR: video window unavailable.\n");

  /* kbinding might open an error dialog (double keymapping), which produces a segfault,
   * when done before the video_window_init(). */
  gui->kbindings = kbindings_new (gui, gui->keymap_file);
  gui->kbindings_enabled = !!gui->kbindings;
  gui->kbindings_enabled |= xitk_bitmove (xine_config_register_bool (gui->xine, "gui.kbindings.implicit_aliases", 0,
    _("Convenient key bindings"),
    _("If not assigned otherwise, \"Ctrl+Key\", \"Meta+Key\" etc. mean the same as \"Key\" alone."),
    CONFIG_LEVEL_ADV, implicit_aliases_cb, gui), 1, 2);

  gui_current_set_index (gui, GUI_MMK_CURRENT);

  panel_init (gui);
  gui->event_reject = 0;
}

/*
 *
 */
typedef struct {
  gGui_t  *gui;
  char   **session_opts;
  int      start;
} _startup_t;

static void on_start(void *data) {
  _startup_t *startup = (_startup_t *) data;
  gGui_t *gui = startup->gui;

  if (!startup->start)
    gui_display_logo (gui);

  if(startup->session_opts) {
    int i = 0;
    int dummy_session;

    while(startup->session_opts[i])
      (void) session_handle_subopt(startup->session_opts[i++], NULL, &dummy_session);

  }

  if (startup->start)
    gui_action_id (gui, ACTID_PLAY);

  xitk_lock (gui->xitk, 1);
  splash_destroy (gui);
  xitk_lock (gui->xitk, 0);
}

static void on_stop (void *data) {
  gGui_t *gui = data;
  xitk_register_signal_handler (gui->xitk, NULL, gui);
  gui_exit_2 (gui);
}

void gui_run(gGui_t *gui, char **session_opts) {
  int auto_start = 0;

  gui->logo_has_changed++;

  /* this finally shows the panel. */
  panel_init2 (gui->panel);

  /* video control window wants some persistant private data. */
  control_main (XUI_W_INIT, gui);

  /* autoscan playlist  */
  if(gui->autoscan_plugin != NULL) {
    const char *const *autoscan_plugins = xine_get_autoplay_input_plugin_ids (gui->xine);
    int         i;

    if(autoscan_plugins) {

      for(i = 0; autoscan_plugins[i] != NULL; ++i) {

	if(!strcasecmp(autoscan_plugins[i], gui->autoscan_plugin)) {
	  int    num_mrls, j;
	  const char * const *autoplay_mrls = xine_get_autoplay_mrls (gui->xine,
							 gui->autoscan_plugin,
							 &num_mrls);

	  if(autoplay_mrls) {
	    for (j = 0; j < num_mrls; j++)
                gui_playlist_append (gui, autoplay_mrls[j],
				     autoplay_mrls[j], NULL, 0, -1, 0, 0);

	    gui->playlist.cur = 0;
            gui_current_set_index (gui, GUI_MMK_CURRENT);
	  }
	}
      }
    }
  }

  panel_playback_ctrl (gui->panel, gui->playlist.num > 0);

  /* We can't handle signals here, xitk handle this, so
   * give a function callback for this.
   */
  xitk_register_signal_handler (gui->xitk, gui_signal_handler, gui);

  /*
   * event loop
   */

  if (gui->flags & XUI_FLAG_start_lirc) {
    lirc_start (gui);
  }

  if (gui->stdctl_enable)
    stdctl_start (gui);

#ifdef HAVE_READLINE
  start_remote_server (gui);
#endif
  gui->session = init_session (gui);

  if(gui->tvout) {
    int size[2] = {0, 0};

    video_window_get_size (gui->vwin, size, VWIN_SIZE_VISIBLE);
    tvout_set_fullscreen_mode (gui->tvout,
      ((video_window_mode (gui->vwin, TOGGLE_MODE) & WINDOWED_MODE) ? 0 : 1), size[0], size[1]);
  }

  if (gui->flags & (XUI_FLAG_start_play | XUI_FLAG_start_plist | XUI_FLAG_start_vis | XUI_FLAG_start_vo_vis |
    XUI_FLAG_start_fullscr | XUI_FLAG_start_xinerama | XUI_FLAG_start_deint | XUI_FLAG_start_setup |
    XUI_FLAG_start_quit)) {

    /* Popup setup window if there is no config file */
    if (gui->flags & XUI_FLAG_start_setup) {
      xine_config_save (gui->xine, gui->cfg_file);
      gui_action_id (gui, ACTID_SETUP);
    }

    /*  The user wants to hide video window  */
    if (gui->flags & XUI_FLAG_start_vo_vis) {
      if (panel_is_visible (gui->panel) < 2)
        gui_action_id (gui, ACTID_TOGGLE_VISIBLITY);
      /* DXR3 case */
      if (video_window_is_visible (gui->vwin) > 1)
        video_window_set_visibility (gui->vwin, 0);
      else
        xine_port_send_gui_data (gui->vo_port, XINE_GUI_SEND_VIDEOWIN_VISIBLE, (int *)NULL);
    }

    /* The user wants to see in fullscreen mode  */
    if (gui->flags & XUI_FLAG_start_fullscr)
      gui_action_id (gui, ACTID_TOGGLE_FULLSCREEN);

#ifdef HAVE_XINERAMA
    /* The user wants to see in xinerama fullscreen mode  */
    if (gui->flags & XUI_FLAG_start_xinerama)
      gui_action_id (gui, ACTID_TOGGLE_XINERAMA_FULLSCREEN);
#endif

    /* User load a playlist on startup */
    if (gui->flags & XUI_FLAG_start_plist) {
      gui_current_set_index (gui, GUI_MMK_CURRENT);

      if(gui->playlist.num) {
	gui->playlist.cur = 0;
        panel_playback_ctrl (gui->panel, 1);
      }
    }

    if (gui->flags & XUI_FLAG_start_deint) {
      gui->flags |= XUI_FLAG_deint;
      post_deinterlace (gui);
      /* panel_raise_window (gui->panel); */
    }

    /*  The user request "play on start" */
    if (gui->flags & XUI_FLAG_start_play) {
      if ((mediamark_get_current_mrl (gui)) != NULL) {
	auto_start = 1;
      }
    }

    /* Flush actions on start */
    if (gui->flags & XUI_FLAG_start_quit)
      gui->flags &= ~(XUI_FLAG_start_play | XUI_FLAG_start_plist | XUI_FLAG_start_vis | XUI_FLAG_start_vo_vis |
        XUI_FLAG_start_fullscr | XUI_FLAG_start_xinerama | XUI_FLAG_start_deint | XUI_FLAG_start_setup);
  }

  gui->oxine = oxine_init (gui);

  {
    _startup_t  startup;

    startup.gui          = gui;
    startup.session_opts = session_opts;
    startup.start        = auto_start;
    xitk_run (gui->xitk, on_start, &startup, on_stop, gui);
  }

  /* save playlist */
  if(gui->playlist.mmk && gui->playlist.num) {
    char buffer[XITK_PATH_MAX + XITK_NAME_MAX + 1];

    snprintf(buffer, sizeof(buffer), "%s/.xine/xine-ui_old_playlist.tox", xine_get_homedir());
    gui_playlist_save (gui, buffer);
  }

  gui->running = 0;
  deinit_session (&gui->session);
#ifdef HAVE_READLINE
  stop_remote_server (gui);
#endif

  kbindings_save (gui, gui->kbindings, gui->keymap_file);
  kbindings_delete(&gui->kbindings);

  /*
   * Close X display before unloading modules linked against libGL.so
   * https://www.xfree86.org/4.3.0/DRI11.html
   *
   * Do not close the library with dlclose() until after XCloseDisplay() has
   * been called. When libGL.so initializes itself it registers several
   * callbacks functions with Xlib. When XCloseDisplay() is called those
   * callback functions are called. If libGL.so has already been unloaded
   * with dlclose() this will cause a segmentation fault.
   */
  xitk_free(&gui->xitk);
}
